/** @file   tank.cpp
 * @brief   Implementation of Tank - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "tank.h"
#include "redrawqueue.h"
#include "GfxManager.h"
#include "soundsamples.h"
#include "animplayer.h"
#include "settings.h"
#include "playercontroller.h"
#include "www_map.h"
#include "warglobals.h"
#include "player.h"
#include "utils.h"
#include <string>
#include <fstream>
#include "AnimId.h"
#include "GfxId.h"
#include "Resource/TextureBitmaps.rh"
using std::string;
using std::ifstream;
using namespace eng2d;

namespace WeWantWar {

/** Constructor
 */
Tank::Tank( Type t ) :
  PolygonObject(),
  m_canCapture( true ),
  m_pCapturedBy( 0 ),
  m_type( t ),
  m_pMoveSound( 0 )
{
  this->initTank();
  
  m_pMoveSound = new SoundSource( "tankmove", Vec2D(0,0) );
  Sound::createSoundSource( m_pMoveSound );
}



/** Destructor
 */
Tank::~Tank()
{
  /** We do not delete the soundsource here because it belongs to SoundPlayer
   * now.
   */
}


/** Updates this tank
 */
void Tank::update()
{
  if ( m_pCapturedBy != 0 ) {
    this->updateCaptured();
  }
  
  // We rotate the cannon tower
  Polygon* pPoly = this->getPolygon(1);
  for ( int i=0; i < pPoly->vertexCount(); i++ ) {
    Polygon::Vtx v;
    m_towerPoly.getVertex( i, &v );
    v.coords.rotate( m_cannonAngle );
    pPoly->setVertex( i, v );
  }
}



/** Redraws this tank
 */
void Tank::redraw( RedrawQueue* pQueue )
{
  PolygonObject::redraw( pQueue );
}



/** Makes sound
 */
void Tank::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_ATTACK ) {
    Sound::playSample( SMP_TANKSHOOT, false );
  } else if ( id == GameObject::SND_PAIN ) {
    Sound::playSample( SMP_METALHIT, false );
  }
  
}



/** Destroyes this tank
 */
void Tank::kill()
{
  this->state( GameObject::STATE_KILLED );
  this->hidden( true );
  
  // Create the explosion
  float ac = 256.0 / 32.0;
  for (float a=0; a < 255; a+=ac) {
    eng2d::Vec2D spdVec( 0,-7 );
    spdVec.rotate( a );
    Bullet* b = BulletTable::createBullet( this, m_position, Bullet::ECarFragments );

    b->setVelocity( spdVec );
    b->iRange = 110;
    b->iDamage = 10;

    WarGlobals::pBulletManager->spawnBullet( b );
  }
  Sound::playSample(SMP_GRENADE, false);
  const Animation& anim = GameAnims::findAnimation( AnimId::KExplosionGrenade );
  AnimPlayer::spawn( anim, this->position(), 0 );
  
  if ( m_pCapturedBy != 0 ) {
    m_pCapturedBy->setHealth(0);
    m_pCapturedBy->kill();
  }
}



/** Handles the bullet hits
 */
bool Tank::hitByBullet( Bullet* pB )
{
  if ( pB->iType == Bullet::ECarFragments ) {
    // We are immune to the fragments emitted from exploding car.
    return false;
  }
  
  // Check if we're captured.
  GameObject* pWeAre = this;
  if ( m_pCapturedBy != 0 ) {
    pWeAre = m_pCapturedBy;
  }
  if ( pB->iType == Bullet::EGrenade && pB->iOwner == pWeAre ) {
    // We are immune to our own bullets
    return false;
  }
  
  if ( pB->iType == Bullet::ETankBullet && pB->iOwner == this ) {
    return false;
  }
  
  if ( this->getCounter( INVULNERABILITY_COUNTER_INDEX ) >= 0 ) {
    return false;
  }
  
  this->causeDamage( pB );

  ParticleSparks* pP = new ParticleSparks( pB->iPosition, pB->velocity(), 12 );
  WarGlobals::pPartManager->addSystem( pP );
  return true;
}



/** Captures this tank
 */
void Tank::capture( GameObject* pObj )
{
  if ( pObj == 0 && m_pCapturedBy != 0 ) {
    // Our current master is someone (not null) and new master will be null.
    // This means that we must tell the previous master that he no longer is
    // our master.
    m_pCapturedBy->state( GameObject::STATE_LIVING );
    m_pCapturedBy->hidden( false );
    Player* pP = dynamic_cast<Player*>( m_pCapturedBy );
    if ( pP != 0 ) {
      pP->setVehicle( 0 );
    }
    m_pCapturedBy = 0;
    return;
  }
  
  m_pCapturedBy = pObj;
  pObj->state( GameObject::STATE_HIBERNATING );
  pObj->hidden( true );
  pObj->position( this->position() );
  
  Player* pP = dynamic_cast<Player*>( pObj );
  if ( pP != 0 ) {
    pP->setVehicle( this );
  }
  
  this->setCounter( NOEXIT_COUNTER_INDEX, 60 );
}



/** Sets the capture-flag.
 */
void Tank::setCapture( bool cap )
{
  m_canCapture = cap;
}



/** Rotates the cannon tower.
 */
void Tank::rotateCannonTower( int a )
{
  m_cannonAngle += a;
  m_cannonAngle = m_cannonAngle & 255;
}



/** Resurrects us
 */
void Tank::resurrect()
{
  this->state( GameObject::STATE_LIVING );
  this->setHealth( 100 );
  this->setCounter( INVULNERABILITY_COUNTER_INDEX, 40 );
  this->hidden( false );
}



/** Sets the forcevector
 */
void Tank::setForce( const Vec2D& rF )
{
}



/** Tells if we're reloading.
 */
bool Tank::reloading() const
{
  if ( this->getCounter( RELOADING_COUNTER_INDEX ) < 0 ) {
    return false;
  }
  return true;
}



/** Returns the type of this object
 */
ObjectID::Type Tank::objectType() const
{
  return ObjectID::TYPE_TANK;
}



/** Tells if this tank can be captured.
 */
bool Tank::canBeCaptured() const
{
  return m_canCapture;
}



/** Returns the direction of the cannon tower
 */
int Tank::cannonAngle() const
{
  return m_cannonAngle;
}



/** Returns the master
 */
GameObject* Tank::getMaster() const
{
  return m_pCapturedBy;
}


/** Initializes the tank
 */
void Tank::initTank()
{
  string polyfile = "data/polygons/tank1.txt";
  
  ifstream fin( polyfile.c_str() );
  if ( !fin ) {
    alert("failed to read tank",0,0, "ok",0,0,0);
    return;
  } else {
    string tmp;
    fin >> tmp;
    int ret = this->read( fin );
    if ( ret != 0 ) {
      alert("failed to read tank",0,0, "ok",0,0,0);
    }
    fin.close();
  }
  
  this->addTexture( GfxManager::findBitmap( GfxId::KTextureBitmaps,
                                            Textures_TankBottom) );
  this->addTexture( GfxManager::findBitmap( GfxId::KTextureBitmaps,
                                            Textures_TankTower) );  
  
  ObjectID::Type oid = ObjectID::TYPE_TANK;
  this->setArmor( Settings::floatObjProp( oid, "tank1_armor:" )  );
  this->boundingSphere( Settings::floatObjProp( oid, "tank1_bsphere:" ) );
  m_speed = Settings::floatObjProp( oid, "tank1_spd:" );
  m_reloadDelay = Settings::intObjProp( oid, "tank1_rdelay:" );
  
  // Set the collisionpoints
  this->setCollisionPoint( 0, Vec2D(-20,-25) );
  this->setCollisionPoint( 1, Vec2D( 20,-25) );
  this->setCollisionPoint( 2, Vec2D( 20, 25) );
  this->setCollisionPoint( 3, Vec2D(-20, 25) );
  this->addCollisionPoint( Vec2D( 0,-25 ) );
  this->addCollisionPoint( Vec2D( 0, 25 ) );
  this->addCollisionPoint( Vec2D(-20, 0 ) );
  this->addCollisionPoint( Vec2D( 20, 0 ) );
  
  // Copy the tower-polygon
  m_towerPoly = *( this->getPolygon(1) );
}



/** Updates the captured tank
 */
void Tank::updateCaptured()
{
  BaseController* pC = m_pCapturedBy->getController();
  if ( pC == 0 ) {
    return;
  }
  
  pC->update();
  if ( pC->left() != 0 ) {
    this->changeAngle( -2 );
    
  } else if ( pC->right() != 0 ) {
    this->changeAngle( 2 );
    
  }
  
  if ( pC->forward() != 0 ) {
    Vec2D mvec( 0, -m_speed );
    mvec.rotate( this->angle() );
    this->move( mvec );
    
  } else if ( pC->backward() != 0 ) {
    Vec2D mvec( 0, -m_speed );
    mvec.rotate(  this->angle() + 128 );
    this->move( mvec );
    
  }
  
  
  // Calculate the angle where the cannon should be heading
  int cx = mouse_x + Map::scrollX;
  int cy = mouse_y + Map::scrollY;
  int d = Utils::findTurningDir( this->position(), Vec2D(cx,cy),
                                 this->m_cannonAngle + this->angle() );
  this->rotateCannonTower( d );

  if ( pC->shoot() != 0 ) {
    this->shoot();
  }
  
  // Set the mouse cursor
  Vec2D mpos( mouse_x, mouse_y );
  WarGlobals::pHud->crosshairPos( mpos );
  
  
  m_pCapturedBy->position( this->position() );
  
  if ( this->getCounter( NOEXIT_COUNTER_INDEX ) < 0 ) {
    if ( key[KEY_E] ) {
      WarGlobals::captureVehicle = true;
    }
  }
  
  // Update the soundsource
  m_pMoveSound->position( this->position() );
  if ( pC->forward() || pC->backward() || pC->left() || pC->right() ) {
    m_pMoveSound->turnOn();
  } else {
    m_pMoveSound->turnOff();
  }
}



/** Shoots
 */
void Tank::shoot()
{
  if ( this->reloading() == true ) {
    return;
  }
  int shootAngle = m_cannonAngle + this->angle();
  Vec2D gunV( 0, -56 );
  gunV.rotate( shootAngle );
  gunV += this->position();
  
  // Create the bullet.
  Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::ETankBullet );
  Vec2D spdV( 0, -(pB->velocity().length()) );
  spdV.rotate( shootAngle );
  pB->setVelocity( spdV );
  
  if ( m_pCapturedBy != 0 ) {
    // If this tank is captured by a gameobject, we set that gameobject to
    // be the shooter.
    pB->iOwner = m_pCapturedBy;
  }
  WarGlobals::pBulletManager->spawnBullet( pB );
  this->makeSound( GameObject::SND_ATTACK );
  const Animation& exploAnim = GameAnims::findAnimation( AnimId::KExplosionSmallWorm );
  AnimPlayer::spawn( exploAnim, gunV, 0 );
  
  this->setCounter( RELOADING_COUNTER_INDEX, m_reloadDelay );
}


} // end of namespace
